<?php

declare(strict_types=1);

use CRM_Pledgeletters_ExtensionUtil as E;

/**
 * Pledge Letters task that reuses the core Print/Merge UI and
 * adds pledge-specific email + date update behavior.
 */
final class CRM_Pledgeletters_Form_Task_PDFLetter extends CRM_Pledge_Form_Task {

  use CRM_Contact_Form_Task_PDFTrait;

  /** @var int[] */
  protected array $pledgeIds = [];

  public function getTemplateFileName(): string {
    return 'CRM/Pledgeletters/Form/Task/PDFLetter.tpl';
  }

  public function preProcess(): void {
    // Build template picker / preview / doc settings
    $this->preProcessPDF();

    // Populate IDs from pledge search selection
    parent::preProcess();
    $this->pledgeIds = array_map('intval', (array) ($this->_pledgeIds ?? []));

    // Support single-mode (?id=)
    $singleId = (int) CRM_Utils_Request::retrieve('id', 'Positive', $this, FALSE);
    if ($singleId && !in_array($singleId, $this->pledgeIds, TRUE)) {
      $this->pledgeIds[] = $singleId;
    }

    if (!$this->pledgeIds) {
      CRM_Core_Error::statusBounce(E::ts('No pledges selected.'));
    }

    $this->assign('selectedPledgeCount', count($this->pledgeIds));
  }

  public function buildQuickForm(): void {
    // Show the full Print/Merge UI (Use Template / Page Format / Preview / WYSIWYG)
    $this->assign('suppressForm', FALSE);
    $this->add('select', 'from_email_address', E::ts('From Email Address'), $this->getFromEmails(), TRUE);
    $this->addPDFElementsToForm();

    // Pledge options
    $this->add('static', 'more_options_header', NULL, E::ts('Pledge Letter Options'));
    // $this->add('checkbox', 'update_receipt', E::ts('Update pledge receipt dates for these pledges'));
    $this->add('checkbox', 'update_thankyou', E::ts('Update pledge thank-you dates for these pledges'));

    // Delivery options
    $emailOptions = [
      'pdf'   => E::ts('Generate PDFs for printing (only)'),
      'email' => E::ts('Send emails where possible. Generate printable PDFs for contacts who cannot receive email.'),
      'both'  => E::ts('Send emails where possible. Generate printable PDFs for all contacts.'),
    ];
    if (CRM_Core_Config::singleton()->doNotAttachPDFReceipt) {
      $emailOptions['pdfemail']      = E::ts('Send emails with an attached PDF where possible. Generate printable PDFs for contacts who cannot receive email.');
      $emailOptions['pdfemail_both'] = E::ts('Send emails with an attached PDF where possible. Generate printable PDFs for all contacts.');
    }
    $this->add('select', 'email_options', E::ts('Print and email options'), $emailOptions, TRUE);

    $this->removeElement('buttons');
    $this->addButtons([
      ['type' => 'upload', 'name' => E::ts('Make Pledge Letters'), 'isDefault' => TRUE],
      ['type' => 'cancel', 'name' => E::ts('Done')],
    ]);

    $this->setDefaults([
      'email_options'   => 'pdf',
      'update_thankyou' => 0,
    ]);
  }

  public function setDefaultValues(): array {
    return $this->getPDFDefaultValues() + [
      'email_options'   => 'pdf',
      'update_thankyou' => 0,
    ];
  }

  public function postProcess(): void {
    $values = $this->controller->exportValues($this->getName());

    [$values, $htmlMessage] = $this->processMessageTemplate($values);
    $messageToken = CRM_Utils_Token::getTokens($htmlMessage);

    $subject       = (string) ($values['activity_subject'] ?? ($values['subject'] ?? E::ts('Pledge Letter')));
    $delivery      = (string) ($values['email_options'] ?? 'pdf');
    $updateTY      = !empty($values['update_thankyou']);
    $campaignId    = (int) ($values['campaign_id'] ?? 0);
    $fromFormatted = CRM_Utils_Mail::formatFromAddress((string) ($values['from_email_address'] ?? ''));

    // Fetch real pledge fields
    $pledgeRows = \Civi\Api4\Pledge::get(FALSE)
      ->addSelect('id', 'contact_id', 'financial_type_id', 'amount', 'currency',
                  'frequency_unit', 'frequency_interval', 'installments',
                  'start_date', 'create_date', 'status_id')
      ->addWhere('id', 'IN', $this->pledgeIds)
      ->setLimit(0)
      ->execute()
      ->getArrayCopy();

    if (!$pledgeRows) {
      CRM_Core_Error::statusBounce(E::ts('No pledges found.'));
    }

    // Compute total_paid / next_pay_* from PledgePayment
    $pledgeIds  = array_column($pledgeRows, 'id');
    $paymentsRS = \Civi\Api4\PledgePayment::get(FALSE)
      ->addSelect('pledge_id', 'scheduled_date', 'scheduled_amount', 'actual_amount', 'status_id:name')
      ->addWhere('pledge_id', 'IN', $pledgeIds)
      ->setLimit(0)
      ->execute();

    $payByPledge = [];
    foreach ($paymentsRS as $pp) {
      $payByPledge[(int) $pp['pledge_id']][] = $pp;
    }
    foreach ($pledgeRows as &$p) {
      $pid      = (int) $p['id'];
      $payments = $payByPledge[$pid] ?? [];
      $totalPaid = 0.0;
      $nextDate  = NULL;
      $nextAmt   = NULL;
      foreach ($payments as $pp) {
        $status   = strtolower((string) ($pp['status_id:name'] ?? ''));
        $schedAmt = (float) ($pp['scheduled_amount'] ?? 0);
        $actual   = (float) ($pp['actual_amount'] ?? 0);
        if ($status === 'completed') {
          $totalPaid += ($actual > 0 ? $actual : $schedAmt);
        } elseif ($status === 'pending') {
          $d = $pp['scheduled_date'] ?? NULL;
          if ($d && ($nextDate === NULL || strtotime($d) < strtotime($nextDate))) {
            $nextDate = $d;
            $nextAmt  = $schedAmt;
          }
        }
      }
      $p['total_paid']      = $totalPaid;
      $p['next_pay_date']   = $nextDate;
      $p['next_pay_amount'] = $nextAmt;
    }
    unset($p);

    // Bucket by contact
    $byContact = [];
    foreach ($pledgeRows as $p) {
      $cid = (int) $p['contact_id'];
      $byContact[$cid]['contact_id'] = $cid;
      $byContact[$cid]['pledges'][]  = $p;
    }

    $emailed = 0;
    $nowSql  = date('YmdHis');
    $attach  = in_array($delivery, ['pdfemail', 'pdfemail_both'], TRUE);
    $pages   = [];

    foreach ($byContact as $cid => $bucket) {
      $pageHtml = $this->renderContactPage($cid, $bucket['pledges'], $htmlMessage, $messageToken);

      // Email if requested & eligible
      $includeInPdf = TRUE;
      if ($this->isLiveMode() && in_array($delivery, ['email', 'both', 'pdfemail', 'pdfemail_both'], TRUE) && $this->canEmail($cid)) {
        $mail = [
          'from'    => $fromFormatted,
          'toEmail' => CRM_Contact_BAO_Contact::getPrimaryEmail($cid),
          'toName'  => CRM_Contact_BAO_Contact::displayName($cid),
          'subject' => $subject ?: E::ts('Pledge Letter'),
          'text'    => '',
          'html'    => $attach ? E::ts('Please see attached.') : $pageHtml,
        ];
        if ($attach) {
          // Generate PDF bytes and attach
          $pdfBytes = CRM_Utils_PDF_Utils::html2pdf([$pageHtml], 'PledgeLetter.pdf', TRUE, $values);
          $tmpFile  = CRM_Utils_File::tempnam('pledgeletter_', '.pdf');
          file_put_contents($tmpFile, $pdfBytes);
          $mail['attachments'] = [[
            'mime_type' => 'application/pdf',
            'name'      => 'PledgeLetter.pdf',
            'fullPath'  => $tmpFile,
          ]];
        }
        try {
          if (CRM_Utils_Mail::send($mail)) {
            $emailed++;
            $includeInPdf = in_array($delivery, ['both', 'pdfemail_both'], TRUE);
          }
        } catch (\Throwable $e) {
            // Fall back to including in the combined PDF
          $includeInPdf = TRUE;
        }
      }

      if ($includeInPdf) {
        $pages[] = $pageHtml;
      }

      if ($this->isLiveMode() && $updateTY) {
        $records = [];
        foreach ($bucket['pledges'] as $pp) {
          $records[] = ['id' => (int) $pp['id'], 'thankyou_date' => $nowSql];
        }
        if ($records) {
          \Civi\Api4\Pledge::save(FALSE)->setRecords($records)->execute();
        }
      }

      // Log an activity
      if ($this->isLiveMode()) {
        $sourceCid = CRM_Core_Session::singleton()->getLoggedInContactID() ?? $cid;
        \Civi\Api4\Activity::create(FALSE)
          ->addValue('activity_type_id:name', 'Print PDF Letter')
          ->addValue('subject', $subject)
          ->addValue('target_contact_id', $cid)
          ->addValue('campaign_id', $campaignId ?: null)
          ->addValue('source_contact_id', $sourceCid)
          ->addValue('details', E::ts('Pledge letter generated'))
          ->execute();
      }
    }

    // Stream combined PDF (if any pages queued)
    if ($pages) {
      CRM_Utils_PDF_Utils::html2pdf($pages, 'PledgeLetters.pdf', FALSE, $values);
      CRM_Utils_System::civiExit();
    }

    // Emails-only path
    CRM_Core_Session::setStatus(E::ts('Emailed %1 contacts.', [1 => $emailed]), '', 'info');
  }

  /**
   * Render HTML for one contact using the same pipeline as preview/editor,
   * and expose {$pledge.*} via Smarty.
   */
  private function renderContactPage(int $contactId, array $pledges, string $htmlMessage, array $messageToken): string {
    $p       = $pledges[0] ?? [];
    $balance = (float) ($p['amount'] ?? 0) - (float) ($p['total_paid'] ?? 0);
    $smarty  = CRM_Core_Smarty::singleton();
    $smarty->assign('pledge', [
      'id'              => $p['id'] ?? NULL,
      'amount'          => CRM_Utils_Money::format((float) ($p['amount'] ?? 0)),
      'total_paid'      => CRM_Utils_Money::format((float) ($p['total_paid'] ?? 0)),
      'balance'         => CRM_Utils_Money::format($balance),
      'next_pay_date'   => CRM_Utils_Date::customFormat($p['next_pay_date'] ?? NULL),
      'next_pay_amount' => CRM_Utils_Money::format((float) ($p['next_pay_amount'] ?? 0)),
    ]);

    $tp = new \Civi\Token\TokenProcessor(\Civi::dispatcher(), [
      'schema'    => ['contactId'],
      'smarty'    => (defined('CIVICRM_MAIL_SMARTY') && CIVICRM_MAIL_SMARTY),
      'contactId' => $contactId,
    ]);
    $tp->addMessage('body', $htmlMessage, 'text/html');
    $row = $tp->addRow(['contactId' => $contactId]);
    $tp->evaluate();

    return $row->render('body');
  }

  /**
   * Determine if we may email the contact.
   */
  private function canEmail(int $cid): bool {
    $email = CRM_Contact_BAO_Contact::getPrimaryEmail($cid);
    if (!$email) {
      return FALSE;
    }
    $contact = civicrm_api3('Contact', 'getsingle', [
      'id'     => $cid,
      'return' => ['is_deceased', 'do_not_email'],
    ]);
    if (!empty($contact['is_deceased']) || !empty($contact['do_not_email'])) {
      return FALSE;
    }
    $onHold = civicrm_api3('Email', 'getcount', [
      'contact_id' => $cid,
      'is_primary' => 1,
      'on_hold'    => 1,
    ]);
    return $onHold == 0;
  }
}
