<?php

require_once 'pledgeletters.civix.php';

use CRM_Pledgeletters_ExtensionUtil as E;

/**
 * Implements hook_civicrm_config().
 *
 * @link https://docs.civicrm.org/dev/en/latest/hooks/hook_civicrm_config/
 */
function pledgeletters_civicrm_config(&$config): void {
  _pledgeletters_civix_civicrm_config($config);
}

/**
 * Implements hook_civicrm_install().
 *
 * @link https://docs.civicrm.org/dev/en/latest/hooks/hook_civicrm_install
 */
function pledgeletters_civicrm_install(): void {
  _pledgeletters_civix_civicrm_install();
}

/**
 * Implements hook_civicrm_enable().
 *
 * @link https://docs.civicrm.org/dev/en/latest/hooks/hook_civicrm_enable
 */
function pledgeletters_civicrm_enable(): void {
  _pledgeletters_civix_civicrm_enable();
}

/**
 * Add a new Action to the Pledge search "Actions" dropdown.
 */
function pledgeletters_civicrm_searchTasks(string $objectType, array &$tasks): void {
  if ($objectType !== 'pledge') {
    return;
  }
  $tasks[] = [
    'title' => E::ts('Pledge Letters - print or email'),
    'class' => 'CRM_Pledgeletters_Form_Task_PDFLetter',
    'result' => FALSE,
    'is_single_mode' => TRUE,
    'weight' => 45,
    'permissions' => ['access CiviCRM', 'access CiviPledge'],
    'url'  => 'civicrm/pledge/task?reset=1&task=letter',
    'key'  => 'letter',
    'name' => E::ts('Send Letter'),
    'title_single_mode' => E::ts('Pledge Letters - print or email'),
  ];
}

/**
 * Register tokens for the picker.
 */
function pledgeletters_civicrm_tokens(array &$tokens): void {
  $tokens['pledge'] = array_merge($tokens['pledge'] ?? [], [
    'pledge.pledge_balance'         => E::ts('Pledge Balance'),
    'pledge.pledge_financial_type'  => E::ts('Pledge Financial Type'),
    'pledge.id'                     => E::ts('Pledge ID'),
    'pledge.pledge_amount'          => E::ts('Total pledge amount'),
    'pledge.create_date'            => E::ts('Pledge Start Date'),
    'pledge.pledge_next_pay_date'   => E::ts('Upcoming pledge payment date'),
    'pledge.pledge_next_pay_amount' => E::ts('Upcoming pledge payment amount'),
    'pledge.pledge_total_paid'      => E::ts('Total pledge payments'),
  ]);
}

/** 
 * Provide values for {pledge.*} using latest pledge per contact 
 */
function pledgeletters_civicrm_tokenValues(
  &$values,
  $contactIDs,
  $job,
  $tokens,
  $context
): void {
  if (empty($tokens['pledge']) || empty($contactIDs)) {
    return;
  }
  $need = array_flip($tokens['pledge']); // e.g. ['pledge_amount' => true, …]

  // Get real pledge fields
  $pledges = \Civi\Api4\Pledge::get(FALSE)
    ->addSelect('id', 'contact_id', 'financial_type_id', 'amount', 'currency',
                'start_date', 'create_date', 'status_id')
    ->addWhere('contact_id', 'IN', $contactIDs)
    ->addOrderBy('id', 'DESC')
    ->setLimit(0)
    ->execute();

  // Keep latest pledge per contact
  $latest = [];
  foreach ($pledges as $p) {
    $cid = (int) $p['contact_id'];
    if (!isset($latest[$cid])) {
      $latest[$cid] = $p;
    }
  }

  // Gather PledgePayment to compute totals / next due
  $pledgeIds = array_column($latest, 'id');
  if ($pledgeIds) {
    $ppRS = \Civi\Api4\PledgePayment::get(FALSE)
      ->addSelect('pledge_id', 'scheduled_date', 'scheduled_amount', 'actual_amount', 'status_id:name')
      ->addWhere('pledge_id', 'IN', $pledgeIds)
      ->setLimit(0)
      ->execute();

    $byPledge = [];
    foreach ($ppRS as $pp) {
      $byPledge[(int) $pp['pledge_id']][] = $pp;
    }

    foreach ($latest as &$p) {
      $pid = (int) $p['id'];
      $payments = $byPledge[$pid] ?? [];
      $paid = 0.0; $nextDate = NULL; $nextAmt = NULL;
      foreach ($payments as $pp) {
        $status   = strtolower((string) ($pp['status_id:name'] ?? ''));
        $schedAmt = (float) ($pp['scheduled_amount'] ?? 0);
        $actual   = (float) ($pp['actual_amount'] ?? 0);
        if ($status === 'completed') {
          $paid += ($actual > 0 ? $actual : $schedAmt);
        } elseif ($status === 'pending') {
          $d = $pp['scheduled_date'] ?? NULL;
          if ($d && ($nextDate === NULL || strtotime($d) < strtotime($nextDate))) {
            $nextDate = $d;
            $nextAmt  = $schedAmt;
          }
        }
      }
      $p['__total_paid']      = $paid;
      $p['__next_pay_date']   = $nextDate;
      $p['__next_pay_amount'] = $nextAmt;
    }
    unset($p);
  }

  foreach ($contactIDs as $cid) {
    $p = $latest[$cid] ?? NULL;
    if (!$p) {
      continue;
    }

    $amount     = (float) ($p['amount'] ?? 0);
    $paid       = (float) ($p['__total_paid'] ?? 0);
    $balanceVal = $amount - $paid;

    $out = [];
    if (isset($need['id'])) {
      $out['pledge.id'] = (int) $p['id'];
    }
    if (isset($need['pledge_amount'])) {
      $out['pledge.pledge_amount'] = CRM_Utils_Money::format($amount);
    }
    if (isset($need['pledge_total_paid'])) {
      $out['pledge.pledge_total_paid'] = CRM_Utils_Money::format($paid);
    }
    if (isset($need['pledge_balance'])) {
      $out['pledge.pledge_balance'] = CRM_Utils_Money::format($balanceVal);
    }
    if (isset($need['create_date'])) {
      // Use start_date for "Pledge Start Date"; change if you prefer create_date
      $out['pledge.create_date'] = CRM_Utils_Date::customFormat($p['start_date'] ?? NULL);
    }
    if (isset($need['pledge_next_pay_date'])) {
      $out['pledge.pledge_next_pay_date'] = CRM_Utils_Date::customFormat($p['__next_pay_date'] ?? NULL);
    }
    if (isset($need['pledge_next_pay_amount'])) {
      $out['pledge.pledge_next_pay_amount'] = CRM_Utils_Money::format((float) ($p['__next_pay_amount'] ?? 0));
    }
    if (isset($need['pledge_financial_type'])) {
      $ftName = '';
      if (!empty($p['financial_type_id'])) {
        $financialTypes = civicrm_api4('FinancialType', 'get', [
          'select' => [
            'name',
          ],
          'where' => [
            ['id', '=', (int) $p['financial_type_id']],
          ],
          'checkPermissions' => FALSE,
        ]);
        $ftName = $financialTypes[0]['name'] ?? '';
      }
      $out['pledge.pledge_financial_type'] = $ftName;
    }

    // Merge into output for this contact
    $values[$cid] = array_merge($values[$cid] ?? [], $out);
  }
}
